
import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { User, Job } from '../types';
import { verificationService } from '../services/api';
import { 
  TrendingUp, 
  ShieldCheck, 
  Mail, 
  Zap, 
  Clock, 
  ExternalLink,
  ChevronRight
} from 'lucide-react';

interface DashboardProps {
  user: User | null;
}

const Dashboard: React.FC<DashboardProps> = ({ user }) => {
  const navigate = useNavigate();
  const [credits, setCredits] = useState(user?.credits || 0);
  const [recentJobs, setRecentJobs] = useState<any[]>([]);

  useEffect(() => {
    const fetchDashboardData = async () => {
      try {
        const liveCredits = await verificationService.getCredits();
        setCredits(liveCredits);
        
        const history = await verificationService.getHistory();
        setRecentJobs(history.slice(0, 3));
      } catch (err) {
        console.error('Error fetching dashboard data');
      }
    };
    fetchDashboardData();
  }, []);

  const stats = [
    { label: 'Credits Remaining', value: credits.toLocaleString(), icon: <Zap className="text-yellow-600" />, subtext: 'Refills on 1st of month', color: 'bg-yellow-50' },
    { label: 'Verified Today', value: user?.verifiedToday?.toLocaleString() || '0', icon: <Mail className="text-indigo-600" />, subtext: '+12% from yesterday', color: 'bg-indigo-50' },
    { label: 'Avg. Confidence', value: '94.2%', icon: <TrendingUp className="text-emerald-600" />, subtext: 'Across last 10k leads', color: 'bg-emerald-50' },
    { label: 'Deliverability Score', value: '98/100', icon: <ShieldCheck className="text-blue-600" />, subtext: 'Safe for sending', color: 'bg-blue-50' },
  ];

  return (
    <div className="space-y-8 animate-in fade-in duration-500">
      <div>
        <h1 className="text-3xl font-bold text-slate-900">Welcome back, {user?.email?.split('@')[0]}</h1>
        <p className="text-slate-500 mt-1">Here's an overview of your email verification activity.</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {stats.map((stat, idx) => (
          <div key={idx} className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
              <div className={`${stat.color} p-2.5 rounded-xl`}>
                {stat.icon}
              </div>
              <span className="text-xs font-semibold text-slate-400 bg-slate-100 px-2 py-1 rounded-md">Live</span>
            </div>
            <h3 className="text-sm font-medium text-slate-500">{stat.label}</h3>
            <p className="text-2xl font-bold text-slate-900 mt-1">{stat.value}</p>
            <p className="text-xs text-slate-400 mt-2 flex items-center">
              <Clock size={12} className="mr-1" />
              {stat.subtext}
            </p>
          </div>
        ))}
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <div className="lg:col-span-2 bg-white rounded-2xl border border-slate-200 shadow-sm overflow-hidden">
          <div className="p-6 border-b border-slate-100 flex items-center justify-between">
            <h2 className="text-lg font-bold text-slate-900">Recent Verification Jobs</h2>
            <button onClick={() => navigate('/history')} className="text-indigo-600 text-sm font-semibold flex items-center hover:text-indigo-700">
              View All <ChevronRight size={16} />
            </button>
          </div>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-slate-50 border-b border-slate-100">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-slate-500 uppercase tracking-wider">Job ID</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-slate-500 uppercase tracking-wider">Total Emails</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-slate-500 uppercase tracking-wider">Status</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-slate-500 uppercase tracking-wider">Date</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-slate-500 uppercase tracking-wider"></th>
                </tr>
              </thead>
              <tbody className="divide-y divide-slate-100">
                {recentJobs.length > 0 ? recentJobs.map((job, i) => (
                  <tr key={i} className="hover:bg-slate-50/50 transition-colors">
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-slate-900">#{job.id}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-600">{job.total}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm">
                      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${job.status === 'completed' ? 'bg-emerald-100 text-emerald-800' : 'bg-amber-100 text-amber-800'}`}>
                        {job.status.charAt(0).toUpperCase() + job.status.slice(1)}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-500">{job.date || 'Today'}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-right text-sm">
                      <button 
                        onClick={() => navigate(`/results/${job.id}`)}
                        className="text-indigo-600 hover:text-indigo-900 font-medium"
                      >
                        Details
                      </button>
                    </td>
                  </tr>
                )) : (
                  <tr>
                    <td colSpan={5} className="px-6 py-8 text-center text-slate-400 text-sm italic">
                      No jobs processed yet. Start by verifying leads.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>

        <div className="bg-indigo-600 rounded-2xl p-8 text-white flex flex-col justify-between relative overflow-hidden group">
          <div className="absolute -right-12 -top-12 w-48 h-48 bg-white/10 rounded-full blur-3xl group-hover:scale-110 transition-transform duration-500"></div>
          <div>
            <div className="bg-white/20 p-3 rounded-xl w-fit mb-6">
              <Zap className="text-white" />
            </div>
            <h2 className="text-2xl font-bold mb-3">Bulk Verify Emails</h2>
            <p className="text-indigo-100 text-sm mb-8 leading-relaxed">
              Upload your CSV file or paste list of emails. Our AI-powered engine performs deep SMTP handshaking to ensure 99%+ deliverability.
            </p>
          </div>
          <button 
            onClick={() => navigate('/verify')}
            className="w-full bg-white text-indigo-600 font-bold py-3.5 rounded-xl hover:bg-slate-50 transition-colors shadow-lg shadow-indigo-900/20"
          >
            Start Verifying
          </button>
        </div>
      </div>

      <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm flex items-center justify-between">
        <div className="flex items-center space-x-4">
          <div className="bg-amber-100 p-3 rounded-full text-amber-600">
            <Clock size={24} />
          </div>
          <div>
            <h3 className="text-lg font-bold text-slate-900">Upgrade to Pro</h3>
            <p className="text-slate-500 text-sm">Unlock higher rate limits and real-time Webhook integrations.</p>
          </div>
        </div>
        <button 
          onClick={() => navigate('/billing')}
          className="px-6 py-2.5 bg-slate-900 text-white rounded-xl font-semibold text-sm hover:bg-slate-800 transition-colors"
        >
          View Plans
        </button>
      </div>
    </div>
  );
};

export default Dashboard;
