
import React, { useState } from 'react';
import { 
  Check, 
  CreditCard, 
  Bitcoin, 
  Zap, 
  ShieldCheck, 
  Globe, 
  ChevronRight,
  TrendingUp,
  Clock,
  RefreshCw,
  AlertCircle
} from 'lucide-react';
import { User } from '../types';
import { billingService } from '../services/api';

interface BillingProps {
  user: User | null;
}

const Billing: React.FC<BillingProps> = ({ user }) => {
  const [loadingPlan, setLoadingPlan] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);

  const handlePurchase = async (planId: string, method: 'stripe' | 'crypto' = 'stripe') => {
    if (planId === 'enterprise') {
      window.location.href = 'mailto:sales@mailvity.cc';
      return;
    }

    try {
      setLoadingPlan(planId);
      setError(null);
      const { url } = await billingService.initiatePayment(planId, method);
      if (url) {
        window.location.href = url;
      }
    } catch (err: any) {
      setError(err.response?.data?.error || 'Failed to initiate checkout. Please try again.');
    } finally {
      setLoadingPlan(null);
    }
  };

  const plans = [
    {
      id: 'starter',
      name: 'Starter',
      price: '$49',
      credits: '10,000',
      description: 'Perfect for small lead lists.',
      features: ['99.9% SMTP Accuracy', 'CSV Export', 'Basic Support', 'Manual Uploads'],
      button: 'Choose Starter',
      popular: false,
    },
    {
      id: 'pro',
      name: 'Professional',
      price: '$129',
      credits: '50,000',
      description: 'Ideal for growing sales teams.',
      features: ['All Starter Features', 'AI Confidence Scoring', 'API Access', 'Priority Support', 'Daily Backups'],
      button: 'Upgrade to Pro',
      popular: true,
    },
    {
      id: 'enterprise',
      name: 'Enterprise',
      price: '$499',
      credits: '500,000',
      description: 'Full-scale email verification.',
      features: ['All Pro Features', 'Custom Webhooks', 'Dedicated Support', 'SLA Guarantee', 'Bulk Discounts'],
      button: 'Contact Sales',
      popular: false,
    }
  ];

  return (
    <div className="space-y-12 animate-in fade-in duration-500">
      <div className="text-center max-w-2xl mx-auto space-y-4">
        <h1 className="text-4xl font-extrabold text-slate-900 tracking-tight">Simple, Transparent Pricing</h1>
        <p className="text-slate-500 text-lg">Verify more leads and scale your outreach with precision validation.</p>
        
        <div className="inline-flex items-center bg-indigo-100 px-4 py-1.5 rounded-full text-indigo-700 font-bold text-xs uppercase tracking-widest">
          <Zap size={14} className="mr-1.5" />
          1 Credit = 1 Verified Email
        </div>
      </div>

      {error && (
        <div className="max-w-md mx-auto p-4 bg-red-50 border border-red-100 rounded-2xl flex items-center space-x-3 text-red-600 text-sm font-medium">
          <AlertCircle size={20} />
          <span>{error}</span>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
        {plans.map((plan, i) => (
          <div 
            key={i} 
            className={`relative flex flex-col bg-white p-8 rounded-3xl border transition-all ${
              plan.popular ? 'border-indigo-600 shadow-xl shadow-indigo-100' : 'border-slate-200 shadow-sm'
            }`}
          >
            {plan.popular && (
              <div className="absolute -top-4 left-1/2 -translate-x-1/2 bg-indigo-600 text-white px-4 py-1 rounded-full text-xs font-bold uppercase tracking-wider shadow-lg">
                Most Popular
              </div>
            )}
            
            <div className="mb-8">
              <h3 className="text-xl font-bold text-slate-900">{plan.name}</h3>
              <p className="text-slate-500 text-sm mt-1">{plan.description}</p>
              <div className="mt-4 flex items-baseline">
                <span className="text-4xl font-extrabold text-slate-900">{plan.price}</span>
                <span className="text-slate-400 ml-1 text-sm">/month</span>
              </div>
              <p className="mt-2 text-indigo-600 font-bold">{plan.credits} Credits</p>
            </div>

            <ul className="flex-1 space-y-4 mb-8">
              {plan.features.map((feature, j) => (
                <li key={j} className="flex items-start text-sm text-slate-600">
                  <Check size={18} className="text-indigo-500 mr-2 shrink-0" />
                  <span>{feature}</span>
                </li>
              ))}
            </ul>

            <button 
              onClick={() => handlePurchase(plan.id, 'stripe')}
              disabled={loadingPlan !== null}
              className={`w-full py-4 rounded-xl font-bold text-sm transition-all flex items-center justify-center space-x-2 ${
                plan.popular ? 'bg-indigo-600 text-white hover:bg-indigo-700 shadow-lg shadow-indigo-200' : 'bg-slate-900 text-white hover:bg-slate-800'
              } disabled:opacity-50`}
            >
              {loadingPlan === plan.id ? (
                <RefreshCw size={18} className="animate-spin" />
              ) : (
                <>
                  <CreditCard size={18} />
                  <span>{plan.button}</span>
                </>
              )}
            </button>
            
            {plan.id !== 'enterprise' && (
              <button 
                onClick={() => handlePurchase(plan.id, 'crypto')}
                disabled={loadingPlan !== null}
                className="mt-3 w-full py-3 border border-slate-200 rounded-xl text-slate-600 font-bold text-xs hover:bg-slate-50 flex items-center justify-center space-x-2 transition-colors"
              >
                <Bitcoin size={14} className="text-amber-500" />
                <span>Pay with Crypto</span>
              </button>
            )}
          </div>
        ))}
      </div>

      <div className="bg-slate-900 rounded-[2.5rem] p-12 text-white overflow-hidden relative">
        <div className="absolute top-0 right-0 w-1/3 h-full bg-indigo-500/10 blur-[100px] pointer-events-none"></div>
        <div className="relative z-10 grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
          <div>
            <h2 className="text-3xl font-bold mb-4">Pay with Crypto or Card</h2>
            <p className="text-slate-400 leading-relaxed mb-8">
              We support modern payment methods including Stripe for credit cards and direct crypto payments (BTC, ETH, USDC) for your privacy and convenience.
            </p>
            <div className="flex flex-wrap gap-4">
              <div className="flex items-center space-x-3 bg-white/5 border border-white/10 px-6 py-3 rounded-2xl">
                <CreditCard className="text-indigo-400" />
                <span className="font-semibold">Visa / Mastercard</span>
              </div>
              <div className="flex items-center space-x-3 bg-white/5 border border-white/10 px-6 py-3 rounded-2xl">
                <Bitcoin className="text-amber-500" />
                <span className="font-semibold">Crypto Payments</span>
              </div>
              <div className="flex items-center space-x-3 bg-white/5 border border-white/10 px-6 py-3 rounded-2xl">
                <Globe className="text-emerald-400" />
                <span className="font-semibold">Stripe Checkout</span>
              </div>
            </div>
          </div>
          
          <div className="bg-white/5 border border-white/10 p-8 rounded-3xl backdrop-blur-sm">
            <h4 className="font-bold text-xl mb-6">Need custom volume?</h4>
            <div className="space-y-4">
              <div className="flex justify-between items-center text-sm">
                <span className="text-slate-400">Custom Credits</span>
                <span className="font-bold">Contact Us</span>
              </div>
              <div className="h-px bg-white/10 w-full" />
              <div className="flex justify-between items-center text-sm">
                <span className="text-slate-400">White Label Access</span>
                <span className="font-bold">Enterprise Only</span>
              </div>
              <div className="h-px bg-white/10 w-full" />
              <p className="text-xs text-slate-500 mt-4 italic">
                All purchases are final. Credits remain valid for 365 days from the date of purchase.
              </p>
              <button 
                onClick={() => window.location.href = 'mailto:sales@mailvity.cc'}
                className="w-full bg-indigo-500 py-3 rounded-xl font-bold mt-4 hover:bg-indigo-600 transition-colors"
              >
                Contact for Bulk Rates
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Billing;
