
<?php
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);
$email = $data['email'] ?? '';
$password = $data['password'] ?? '';

if (empty($email) || empty($password)) {
    http_response_code(400);
    echo json_encode(['message' => 'Email and password are required.']);
    exit;
}

try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    if (!$user || !password_verify($password, $user['password_hash'])) {
        http_response_code(401);
        echo json_encode(['message' => 'Invalid email or password.']);
        exit;
    }

    if ($user['status'] === 'suspended') {
        http_response_code(403);
        echo json_encode(['message' => 'Account is suspended. Please contact support.']);
        exit;
    }

    // Get count of verifications today for this user
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM verification_results r JOIN verification_jobs j ON r.job_id = j.id WHERE j.user_id = ? AND DATE(r.created_at) = CURDATE()");
    $stmt->execute([$user['id']]);
    $verifiedToday = $stmt->fetchColumn();

    // Generate JWT
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'id' => (int)$user['id'],
        'email' => $user['email'],
        'role' => $user['role'],
        'isAdmin' => ($user['role'] === 'admin'),
        'exp' => time() + (3600 * 24)
    ]);

    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;

    echo json_encode([
        'token' => $jwt,
        'user' => [
            'id' => $user['id'],
            'email' => $user['email'],
            'credits' => (int)$user['credits_balance'],
            'plan' => $user['plan_type'],
            'verifiedToday' => (int)$verifiedToday,
            'role' => $user['role'],
            'status' => $user['status'],
            'isAdmin' => ($user['role'] === 'admin')
        ]
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['message' => 'Database error: ' . $e->getMessage()]);
}
?>
