
<?php
class VerificationEngine {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    public function verifyEmail($email) {
        $score = 0;
        $reason = "";
        $status = "Unknown";

        // 1. Syntax Check
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ["status" => "Invalid", "confidence" => 0, "reason" => "Invalid email syntax"];
        }
        $score += 20;

        // 2. DNS / MX Lookup
        $domain = substr(strrchr($email, "@"), 1);
        $mxHosts = [];
        if (!getmxrr($domain, $mxHosts)) {
            return ["status" => "Invalid", "confidence" => 20, "reason" => "No MX records found for domain"];
        }
        $score += 20;

        // 3. SMTP Handshake
        $mxHost = $mxHosts[0];
        $smtpResult = $this->checkSmtp($mxHost, $email);

        if ($smtpResult['code'] === 250) {
            $score += 30;
            $status = "Valid";
            $reason = "Mailbox accepted via SMTP handshake";
        } elseif ($smtpResult['code'] === 450 || $smtpResult['code'] === 451) {
            $score -= 10;
            $status = "Unknown";
            $reason = "Target server is greylisting; try again later";
        } else {
            $status = "Invalid";
            $reason = "SMTP rejected: " . $smtpResult['message'];
        }

        // 4. Domain Behavior (Catch-all check simulation)
        $stmt = $this->pdo->prepare("SELECT is_catch_all FROM domain_stats WHERE domain = ?");
        $stmt->execute([$domain]);
        $domainData = $stmt->fetch();
        if ($domainData && $domainData['is_catch_all']) {
            $score -= 15;
            $status = "Catch-All";
            $reason .= " (Domain detected as catch-all)";
        }

        // 5. Finalize Score (Clamp 0-100)
        $confidence = max(0, min(100, $score));

        return [
            "email" => $email,
            "status" => $status,
            "confidence" => $confidence,
            "reason" => $reason
        ];
    }

    private function checkSmtp($host, $toEmail) {
        $timeout = 10;
        $errno = 0;
        $errstr = "";
        $conn = fsockopen($host, 25, $errno, $errstr, $timeout);

        if (!$conn) return ['code' => 500, 'message' => "Connection failed: $errstr"];

        // Read greeting
        fgets($conn);

        // HELO
        fputs($conn, "HELO " . $_SERVER['SERVER_NAME'] . "\r\n");
        fgets($conn);

        // MAIL FROM
        fputs($conn, "MAIL FROM:<" . SENDER_EMAIL . ">\r\n");
        fgets($conn);

        // RCPT TO (The actual verification step)
        fputs($conn, "RCPT TO:<$toEmail>\r\n");
        $response = fgets($conn);
        $code = (int)substr($response, 0, 3);

        // QUIT (Very important: do NOT send DATA)
        fputs($conn, "QUIT\r\n");
        fclose($conn);

        return ['code' => $code, 'message' => trim(substr($response, 4))];
    }
}
?>
