
import React, { useState, useEffect } from 'react';
import { Routes, Route, Navigate, useNavigate, useLocation } from 'react-router-dom';
import Sidebar from './components/Sidebar';
import Navbar from './components/Navbar';
import Dashboard from './pages/Dashboard';
import Verify from './pages/Verify';
import Results from './pages/Results';
import Billing from './pages/Billing';
import Admin from './pages/Admin';
import History from './pages/History';
import Login from './pages/Login';
import Register from './pages/Register';
import { AuthState, User } from './types';

const App: React.FC = () => {
  const navigate = useNavigate();
  const location = useLocation();
  const [auth, setAuth] = useState<AuthState>({
    user: null,
    token: localStorage.getItem('token'),
    isAuthenticated: !!localStorage.getItem('token'),
  });

  useEffect(() => {
    const savedUser = localStorage.getItem('user');
    if (savedUser) {
      try {
        setAuth(prev => ({ ...prev, user: JSON.parse(savedUser) }));
      } catch (e) {
        localStorage.removeItem('user');
        localStorage.removeItem('token');
      }
    }
  }, []);

  const handleLogin = (user: User, token: string) => {
    localStorage.setItem('token', token);
    localStorage.setItem('user', JSON.stringify(user));
    setAuth({ user, token, isAuthenticated: true });
    navigate('/dashboard');
  };

  const handleLogout = () => {
    localStorage.removeItem('token');
    localStorage.removeItem('user');
    setAuth({ user: null, token: null, isAuthenticated: false });
    navigate('/login');
  };

  const isAuthPage = ['/login', '/register'].includes(location.pathname);

  if (!auth.isAuthenticated && !isAuthPage) {
    return <Navigate to="/login" replace />;
  }

  return (
    <div className="flex min-h-screen bg-slate-50">
      {auth.isAuthenticated && <Sidebar user={auth.user} onLogout={handleLogout} />}
      
      <div className="flex-1 flex flex-col min-w-0 overflow-hidden">
        {auth.isAuthenticated && <Navbar user={auth.user} />}
        
        <main className="flex-1 overflow-y-auto p-4 md:p-8">
          <Routes>
            <Route path="/login" element={<Login onLogin={handleLogin} />} />
            <Route path="/register" element={<Register onLogin={handleLogin} />} />
            <Route path="/dashboard" element={<Dashboard user={auth.user} />} />
            <Route path="/verify" element={<Verify user={auth.user} />} />
            <Route path="/history" element={<History />} />
            <Route path="/results/:jobId" element={<Results />} />
            <Route path="/billing" element={<Billing user={auth.user} />} />
            <Route 
              path="/admin" 
              element={auth.user?.role === 'admin' ? <Admin /> : <Navigate to="/dashboard" replace />} 
            />
            <Route path="/" element={<Navigate to="/dashboard" replace />} />
            <Route path="*" element={<Navigate to="/dashboard" replace />} />
          </Routes>
        </main>
      </div>
    </div>
  );
};

export default App;
